<?php

namespace App\Services;

use App\Models\ExcelCampaign;
use App\Models\Masters\AgentMaster;
use App\Models\Masters\BranchMaster;
use App\Models\Transactions\BeneficiaryImport;
use App\Models\User;
use Carbon\Carbon;
use Google_Client;
use Google_Service_Sheets;
use Log;
use Google_Service_Sheets_ValueRange;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use stdClass;
use Illuminate\Support\Str;
use DB;

class ExcelCampaignService
{
    protected $client;
    protected $service;
    protected $spreadsheetId;
    protected $commonService;
    protected $sheetName;
    private $branchId;

    const EMPTY_CAMPAIGN_LEAD_ASSIGN_BRANCHS = [5, 6, 15, 17, 23, 24];

    public function __construct(CommonService $commonService)
    {
        $this->commonService = $commonService;
        $this->branchId = [];
    }

    function startFetching()
    {
        try {
            // $this->getWatiFebCampaign();
            $param = $this->getCampaignSheets();
            $this->fetchExcelCampaign($param);
            $this->removeDuplicateLeads();
            // $this->assignNullBranchLeadsToBranch();
            $this->assignLeadsToCallCenter();
            $this->sendNotification();
        } catch (\Exception $e) {
            Log::info('excel campaign fetching error : ' . $e->getMessage());
        }
    }

    function sendNotification()
    {
        if (count($this->branchId) > 0) {

            $playerIds = User::select('users.player_id')
                ->join('agent_master', 'agent_master.id', 'users.reference_id')
                ->where('agent_master.status', AgentMaster::STATUS_ACTIVE)
                ->where('users.user_type', User::USER_TYPE_AGENT)
                ->whereIn('agent_master.branch_id', $this->branchId)
                ->where('agent_master.vendor_campaign_lead_assign', 1)
                ->where('users.player_id', '!=', 'null')
                ->whereNotNull('users.player_id')
                ->pluck('player_id');

            if (count($playerIds) > 0) {
                $url = 'https://api.onesignal.com/notifications?c=push';
                $bodyParams = array(
                    'app_id' => constants('ONE_SIGNAL_APP_ID'),
                    'contents' => array('en' => 'You have received new lead from hyperlocal leads...'),
                    'headings' => array('en' => 'Hyperlocal Lead'),
                    'include_player_ids' => $playerIds
                );
                $response = $this->commonService->coreGuzzlePostOneSignal($url, $bodyParams);
                Log::info('One signal response excel campaign : ' . json_encode($response));
                if ($response->status == 200 && isset($response->data->id) && $response->data->id) {
                    Log::info('One signal notification sent excel campaign ');
                }
            }
        }
    }

    function assignLeadsToCallCenter()
    {
        $today = Carbon::now()->format('Y-m-d');
        $campaign = ExcelCampaign::where('created_at', '>=', $today . ' 00:00:00')
            ->where('created_at', '<=', $today . ' 23:59:59')
            ->whereNull('branch_id')
            ->get();
        if ($campaign) {
            foreach ($campaign as $row) {
                $row->branch_id = 32;
                if ($row->save()) {
                    $mobile = $this->commonService->formatMobileNo($row->phone);
                    $customer = new BeneficiaryImport();
                    $customer->branch_id = 32;
                    $customer->customer_name = $row->name;
                    $customer->mobile_number = $mobile;
                    $customer->invoice_date = Carbon::parse($row->date_created)->format("Y-m-d");
                    $customer->source = BeneficiaryImport::SOURCE_TYPE_EXCEL_CAMPAIGN;
                    $customer->excel_campaign_id = $row->id;
                    $customer->save();
                }
            }
        }
    }

    function removeDuplicateLeads()
    {
        $today = Carbon::now()->format('Y-m-d');
        $campaign = ExcelCampaign::select('id', 'phone', 'branch_id')
            ->where('created_at', '>=', $today . ' 00:00:00')
            ->where('created_at', '<=', $today . ' 23:59:59')
            ->groupBy('phone')
            ->havingRaw('COUNT(*) > 1')
            ->get();
        if ($campaign) {
            foreach ($campaign as $row) {
                $row->delete();
            }
        }
    }

    function assignNullBranchLeadsToBranch()
    {
        $today = Carbon::now()->format('Y-m-d');
        $branches = collect(self::EMPTY_CAMPAIGN_LEAD_ASSIGN_BRANCHS);
        $branchLoad = ExcelCampaign::whereIn('branch_id', $branches)
            ->select('branch_id', DB::raw('COUNT(*) as total'))
            ->where('created_at', '>=', $today . ' 00:00:00')
            ->where('created_at', '<=', $today . ' 23:59:59')
            ->whereNull('deleted_at')
            ->groupBy('branch_id')
            ->pluck('total', 'branch_id');

        // initialize missing branches
        $branches->each(function ($branchId) use (&$branchLoad) {
            if (! $branchLoad->has($branchId)) {
                $branchLoad = $branchLoad->put($branchId, 0);
            }
        });

        // assign leads
        $unassigned = ExcelCampaign::whereNull('branch_id')->get();
        foreach ($unassigned as $lead) {
            // sort branches by load and pick the first key
            $branchId = $branchLoad->sort()->keys()->first();

            $lead->branch_id = $branchId;
            $lead->save();
            if ($lead->id) {
                $mobile = $this->commonService->formatMobileNo($lead->phone);
                $customer = new BeneficiaryImport();
                $customer->branch_id = $branchId;
                $customer->customer_name = $lead->name;
                $customer->mobile_number = $mobile;
                $customer->invoice_date = Carbon::parse($lead->date_created)->format("Y-m-d");
                $customer->source = BeneficiaryImport::SOURCE_TYPE_EXCEL_CAMPAIGN;
                $customer->excel_campaign_id = $lead->id;
                $customer->save();
                if (!in_array($branchId, $this->branchId)) {
                    array_push($this->branchId, $branchId);
                }
            }
            // update the load count
            $branchLoad = $branchLoad->put($branchId, $branchLoad[$branchId] + 1);
        }
    }

    function getCampaignSheets()
    {
        $campaigns = new Collection();
        $items = new stdClass;
        // $items->name = 'TestSheet';
        $items->name = 'CONVO STARTED - REV1';
        $items->branchId = 32;
        $items->columnMapping = array(
            'dateTime' => 0,
            'name' => 1,
            'phone' => 2,
            'devloperRef' => 3,
            'devloperRefColumnName' => 'D',
        );
        $campaigns->add($items);

        // $columnMapping = array(
        //     'dateTime' => 0,
        //     'name' => 1,
        //     'phone' => 2,
        //     'branchCode' => 3,
        //     'devloperRef' => 4,
        //     'devloperRefColumnName' => 'E',
        // );

        // $items = new stdClass;
        // $items->name = 'CHENNAI';
        // $items->branchId = null;
        // $items->columnMapping = $columnMapping;
        // $campaigns->add($items);

        // $items = new stdClass;
        // $items->name = 'TRICHY';
        // $items->branchId = null;
        // $items->columnMapping = $columnMapping;
        // $campaigns->add($items);

        // $items = new stdClass;
        // $items->name = 'MADURAI';
        // $items->branchId = null;
        // $items->columnMapping = $columnMapping;
        // $campaigns->add($items);

        return $campaigns;
    }

    function watiFebSpreadSheetConfig($param)
    {
        $this->spreadsheetId = '1Y_VaN6aAZp_Ddx6MzdWXKF5ufGpnHpUlmra0lnooo7U';
        $this->client = new Google_Client();
        $this->client->setApplicationName('Viveks Google Sheets');
        $this->client->setScopes([Google_Service_Sheets::SPREADSHEETS]);
        $this->client->setAuthConfig(storage_path('app/gsheet-inter-doers-c6f3700275a1.json'));
        $this->client->setAccessType('offline');
        $this->service = new Google_Service_Sheets($this->client);
        $this->sheetName = $param->name;
    }

    public function updateRow($range, $values)
    {
        $body = new Google_Service_Sheets_ValueRange([
            'values' => [$values]
        ]);

        $params = ['valueInputOption' => 'RAW'];

        return $this->service->spreadsheets_values->update(
            $this->spreadsheetId,
            $range,
            $body,
            $params
        );
    }

    function fetchExcelCampaign($param)
    {
        foreach ($param as $branchData) {
            Log::info($branchData->name . ' fetching started ');
            $this->watiFebSpreadSheetConfig($branchData);
            $response = $this->service->spreadsheets_values->get($this->spreadsheetId, $this->sheetName);
            $data = $response->getValues();

            if (!$data) {
                return;
            }

            $processedCount = 0;
            $rowsToUpdate = [];
            $updateValues = [];

            foreach ($data as $key => $row) {
                if ($key == 0 || (isset($row[$branchData->columnMapping['devloperRef']]) && $row[$branchData->columnMapping['devloperRef']] == 'Done')) {
                    continue; // Skip header and already processed rows
                }

                if ($processedCount >= 100) {
                    break; // Stop after 100 records
                }

                // $campaignCount = ExcelCampaign::where('phone', $row[$branchData->columnMapping['phone']])->count();

                // if ($campaignCount > 0) {
                //     $rowsToUpdate[] = $key + 1;
                //     $updateValues[] = ['Done'];
                //     continue;
                // }

                // $branchDet = new stdClass;
                // $branchDet->id = 32;
                $branchCode = null;
                $branchDet = null;

                $stores = collect([$row[5] ?? null, $row[6] ?? null, $row[7] ?? null]);
                $filteredData = $stores->filter(function ($string) {
                    return !empty($string) && !Str::contains($string, '@');
                });

                if ($filteredData && $filteredData->count() > 0) {
                    $branchCode = $filteredData[0] ?? null;
                    if ($branchCode) {
                        $branchDet = BranchMaster::where('excel_campaign_code', $branchCode)->where('status', BranchMaster::STATUS_ACTIVE)->first();
                    }
                }

                $newStores = $row[4] ?? null;
                if (!$branchDet && $newStores) {
                    $branchCode = $newStores;
                    $branchDet = BranchMaster::where('excel_campaign_code', $newStores)->where('status', BranchMaster::STATUS_ACTIVE)->first();
                }

                // if (!isset($branchDet->id)) {
                //     $branchDet = new stdClass;
                //     $branchDet->id = 32;
                //     $branchCode = null;
                // }

                // if ($branchData->branchId > 0) {
                //     $branchDet = new stdClass;
                //     $branchDet->id = $branchData->branchId;
                // } else {
                //     $branchCode = $row[$branchData->columnMapping['branchCode']] ?? null;
                //     if ($branchCode) {
                //         $branchDet = BranchMaster::where('excel_campaign_code', $branchCode)->first();
                //     }
                // }

                $dateCreated = !empty($row[$branchData->columnMapping['dateTime']]) ? date('Y-m-d H:i:s', strtotime($row[$branchData->columnMapping['dateTime']])) : null;

                // Creating a campaign entry
                $campaignData = ExcelCampaign::create([
                    'name' => $row[$branchData->columnMapping['name']] ?? '',
                    'phone' => $row[$branchData->columnMapping['phone']] ?? '',
                    'branch' => $branchCode ?? '',
                    'branch_id' => $branchDet->id ?? null,
                    'date_created' => $dateCreated
                ]);

                if ($campaignData) {
                    $rowsToUpdate[] = $key + 1;
                    $updateValues[] = ['Done'];

                    if (isset($branchDet->id) && !empty($row[$branchData->columnMapping['phone']])) {
                        $mobile = $this->commonService->formatMobileNo($row[$branchData->columnMapping['phone']]);
                        BeneficiaryImport::create([
                            'branch_id' => $branchDet->id,
                            'customer_name' => $campaignData->name,
                            'mobile_number' => $mobile,
                            'invoice_date' => $dateCreated,
                            'source' => BeneficiaryImport::SOURCE_TYPE_EXCEL_CAMPAIGN,
                            'excel_campaign_id' => $campaignData->id,
                        ]);
                        if (!in_array($branchDet->id, $this->branchId)) {
                            array_push($this->branchId, $branchDet->id);
                        }
                    }
                }

                $processedCount++;
            }

            if (!empty($rowsToUpdate)) {
                $dataToUpdate = [];
                foreach ($rowsToUpdate as $index => $rowNum) {
                    $dataToUpdate[] = [
                        'range' => "{$this->sheetName}!" . $branchData->columnMapping['devloperRefColumnName'] . "{$rowNum}",
                        'values' => [['Done']]
                    ];
                }

                $this->batchUpdateRows($dataToUpdate);
            }
            Log::info($branchData->name . ' fetching end ');
        }
    }

    // function getWatiFebCampaign()
    // {
    //     $this->watiFebSpreadSheetConfig();
    //     $response = $this->service->spreadsheets_values->get($this->spreadsheetId, $this->sheetName);
    //     $data = $response->getValues();

    //     if (!$data) {
    //         return;
    //     }

    //     $processedCount = 0;
    //     $rowsToUpdate = [];
    //     $updateValues = [];

    //     foreach ($data as $key => $row) {
    //         if ($key == 0 || (isset($row[7]) && $row[7] == 'Done')) {
    //             continue; // Skip header and already processed rows
    //         }

    //         if ($processedCount >= 100) {
    //             break; // Stop after 100 records
    //         }

    //         $branchCode = $row[3] ?? null;
    //         $branchDet = BranchMaster::where('excel_campaign_code', $branchCode)->first() ??
    //             BranchMaster::where('id', 32)->first();

    //         $dateCreated = !empty($row[0]) ? date('Y-m-d H:i:s', strtotime($row[0])) : null;

    //         // Creating a campaign entry
    //         $campaignData = ExcelCampaign::create([
    //             'name' => $row[1] ?? '',
    //             'phone' => $row[12] ?? '',
    //             'branch' => $branchCode ?? '',
    //             'branch_id' => $branchDet->id ?? null,
    //             'date_created' => $dateCreated
    //         ]);

    //         if ($campaignData) {
    //             $rowsToUpdate[] = $key + 1; // Store row number only
    //             $updateValues[] = ['Done'];

    //             if (isset($branchDet->id) && !empty($row[12])) {
    //                 $mobile = $this->commonService->formatMobileNo($row[12]);
    //                 BeneficiaryImport::create([
    //                     'branch_id' => $branchDet->id,
    //                     'customer_name' => $campaignData->name,
    //                     'mobile_number' => $mobile,
    //                     'invoice_date' => $dateCreated,
    //                     'source' => BeneficiaryImport::SOURCE_TYPE_EXCEL_CAMPAIGN,
    //                     'excel_campaign_id' => $campaignData->id,
    //                 ]);
    //             }
    //         }

    //         $processedCount++;
    //     }

    //     if (!empty($rowsToUpdate)) {
    //         $dataToUpdate = [];
    //         foreach ($rowsToUpdate as $index => $rowNum) {
    //             $dataToUpdate[] = [
    //                 'range' => "{$this->sheetName}!H{$rowNum}",
    //                 'values' => [['Done']]
    //             ];
    //         }

    //         $this->batchUpdateRows($dataToUpdate); // Batch update specific rows
    //     }
    // }

    /**
     * Perform a batch update for multiple rows at once.
     */
    private function batchUpdateRows($updates)
    {
        $requestBody = new \Google\Service\Sheets\BatchUpdateValuesRequest([
            'valueInputOption' => 'RAW',
            'data' => $updates
        ]);

        $this->service->spreadsheets_values->batchUpdate($this->spreadsheetId, $requestBody);
    }


    function getExcelCampaignUserDetails($user)
    {
        if ($user->user_type == User::USER_TYPE_AGENT) {
            $agent = AgentMaster::select('*')->find($user->reference_id);
            if ($agent && in_array($agent->designation, AgentMaster::CAMPAGIN_ROLE_ACCESS_DESIGNATIONS)) {
                return $agent;
            }
        }
    }

    function excelCampaignList(Request $request)
    {
        $item = new stdClass;
        $agent = $this->getExcelCampaignUserDetails($request->user());
        if (!$agent) {
            $item->recordsTotal = 0;
            $item->recordsFiltered = 0;
            $item->data = [];
            return response()->json($item);
        }
        $sql = ExcelCampaign::select(
            'excel_campaign.id',
            'beneficiary_import.customer_name',
            'beneficiary_import.mobile_number'
        )
            ->join('beneficiary_import', 'beneficiary_import.excel_campaign_id', 'excel_campaign.id')
            ->where('excel_campaign.branch_id', $agent->branch_id)
            ->where('beneficiary_import.status', 1)
            ->whereNull('beneficiary_import.agent_id');
        $clone = clone $sql;
        if ($request->page && $request->limit) {
            $limit = $request->page * $request->limit;
            $skip = $limit - $request->limit;
            $sql->skip($skip)->take($limit);
        }
        $sql->orderBy('excel_campaign.id');
        $data = $sql->get();

        $item->recordsTotal = $clone->get()->count();
        $item->recordsFiltered = $data->count();
        $item->data = $data;
        return response()->json($item);
    }

    function assignCampaignLeads(Request $request)
    {
        if (count($request->leadsIds) > 0) {
            BeneficiaryImport::whereIn('excel_campaign_id', $request->leadsIds)->update(['agent_id' => $request->agentId]);
            $return = new stdClass;
            $return->status = 1;
            return $this->commonService->makeSuccessResponse('Success', $return);
        }
        return $this->commonService->makeErrorResponse('failed');
    }
}
