<?php

namespace App\Services;

use App\Models\CRMTickets;
use App\Models\EnquiryHistory;
use App\Models\Feedback;
use App\Models\FeedbackHistory;
use App\Models\FeedbackMaster;
use App\Models\JustDialLeads;
use App\Models\Masters\AgentMapping;
use App\Models\Masters\CallSkipReason;
use App\Models\Masters\ClusterBranchMapping;
use App\Models\Masters\EscalationMatrix;
use App\Models\NumberingSystem;
use App\Models\RoleMaster;
use App\Models\Sales;
use App\Models\Transactions\BeneficiaryImport;
use App\Models\Transactions\CallHistory;
use App\Models\Transactions\ShortLink;
use App\Models\User;
use App\Services\CommonService;
use App\Services\FormService;
use App\Services\Reports\ReportService;
use App\Services\SmsNotificationService;
use App\Utilities\AppClass;
use Carbon\Carbon;
use DB;
use Illuminate\Http\Request as HttpRequest;
use Illuminate\Support\Facades\Auth;
use Log;

class FeedbackService
{


    private $formService;
    private $commonService;
    private $smsNotificationService;
    private $reportService;

    public function __construct(FormService $formService, CommonService $commonService, SmsNotificationService $smsNotificationService, ReportService $reportService)
    {
        $this->formService = $formService;
        $this->commonService = $commonService;
        $this->smsNotificationService = $smsNotificationService;
        $this->reportService = $reportService;
    }

    public function getUserData()
    {
        $users = User::where('users.role', '<>', 0)
            ->join('role_master', 'role_master.id', 'users.role')
            ->select('users.id', 'users.name', 'users.email', 'users.status', 'role_master.description as role')
            ->get();
        return $users;
    }

    public function isNewMaster($id = null)
    {
        if ($id) {
            $id = decrypt($id);
            $feedback = FeedbackMaster::find($id);
            $feedback->route = route('feedback-master.update', $id);
            $feedback->method = 'PUT';
            $feedback->breadcrumbName = 'Edit';
        } else {
            $feedback = new FeedbackMaster();
            $feedback->route = route('feedback-master.store');
            $feedback->method = 'POST';
            $feedback->breadcrumbName = 'Create';
        }

        $feedback->breadcrumbAction = route('feedback-master.index');
        $feedback->breadcrumbTitle = 'Feedback';
        $feedback->modalSuccessUrl = route('feedback-master.index');
        $feedback->modalFailureUrl = route('feedback-master.index');
        return $feedback;
    }

    public function isNew($id = null)
    {
        if ($id) {
            $data = BeneficiaryImport::find($id);
            $data->formType = 'auto-dialer';
        } else {
            $data = new AppClass();
            $data->formType = 'manual';
        }

        $data->route = route('feedback.saveFeedback');
        $data->method = 'POST';
        $data->breadcrumbName = 'Create';
        $data->breadcrumbAction = route('caller.index');
        $data->breadcrumbTitle = 'Import';

        return $data;
    }

    public function getMasterFeedbackForm($request)
    {
        $feedback = FeedbackMaster::where('scheme_id', $request->scheme)->where('program_id', $request->program)->whereStatus(FeedbackMaster::STATUS_ACTIVE)->first();
        $feedback->html = '';
        if ($feedback) {
            $feedback->html = $this->createFeedbackForm($feedback->contents, 'feedback');
        }
        return $feedback;
    }

    public function createFeedbackForm($contents, $type = null)
    {
        $html = '';
        $jsonForm = json_decode($contents, true);

        if ($jsonForm) {
            foreach ($jsonForm as $json) {
                $formField = $this->formService->createFormFields($json, $type);
                $html .= $formField;
            }
        }
        return $html;
    }

    public function saveData(HttpRequest $request)
    {

        $schemeId = $request->has('schemeId') ? $request->get('schemeId') : '';
        $programId = $request->has('programId') ? $request->get('programId') : '';
        $formJson = $request->has('formJson') ? $request->get('formJson') : '';
        $uniqueId = $request->has('uniqueId') ? $request->get('uniqueId') : '';

        $formJson = json_decode($formJson, true);
        $updatedJson = $this->removeHtmlCharFromJson($formJson);

        if ($uniqueId == '') {
            $isExists = FeedbackMaster::where('scheme_id', $schemeId)->where('program_id', $programId)->whereStatus(FeedbackMaster::STATUS_ACTIVE)->whereNull('deleted_at')->count();
            if ($isExists > 0) {
                return 'exists';
            }
        }

        if ($uniqueId) {
            $feedback = FeedbackMaster::find($uniqueId);
            $feedback->updated_by = Auth::user()->id;
        } else {
            $feedback = new FeedbackMaster();
            $feedback->created_by = Auth::user()->id;
        }

        $feedback->scheme_id = $schemeId;
        $feedback->program_id = $programId;
        $feedback->contents = $updatedJson;
        $feedback->save();

        if ($feedback) {
            return 'success';
        }
    }

    public function removeHtmlCharFromJson($json)
    {
        foreach ($json as $key => $value) {
            $json[$key]['name'] = isElementHasValue($value, 'name');
        }
        return json_encode($json);
    }

    public function saveFeedbackChild(HttpRequest $request)
    {
        $data = new BeneficiaryImport();
        $data->customer_name = isset($request->customerName) ? $request->customerName : null;
        $data->mobile_number = $this->commonService->formatMobileNo($request->phone);
        $data->invoice_date = Carbon::parse($request->date)->format('Y-m-d');
        $data->description = isset($request->description) ? $request->description : null;
        $data->branch_id = isset($request->branch) ? $request->branch : null;
        $data->agent_id = isset($request->agent) ? $request->agent : null;
        $data->import_type = BeneficiaryImport::IMPORT_TYPE_WEB_APP;
        $data->source = BeneficiaryImport::SOURCE_TYPE_EXCEL;
        $data->created_by = Auth::user()->id;
        if ($data->save()) {
            return true;
        }
    }

    public function generateFeedbackURL($feedback)
    {
        $baseFeedbackURL = constants('FEEDBACK_SMS_URL');
        $url = $baseFeedbackURL . $feedback->id;
        $tinyUrl = $this->smsNotificationService->createTinyUrl($url);
        $shortLink = new ShortLink();
        $shortLink->code = $feedback->id;
        $shortLink->link = $tinyUrl;
        $shortLink->save();
        return $tinyUrl;
    }

    public function getComplaintsDetailsCore($beneficiary)
    {
        $today = Carbon::now()->format('Y-m-d');
        $sql = DB::table('feedback')
            ->join('beneficiary_import', 'beneficiary_import.id', 'feedback.beneficiary_ref_id')
            ->where('beneficiary_import.district_id', $beneficiary->district_id)
            ->where('beneficiary_import.scheme_id', $beneficiary->scheme_id)
            ->where('beneficiary_import.program_id', $beneficiary->program_id)
            ->whereDate('feedback.created_at', $today)
            ->where('feedback.is_compliant', Feedback::IS_COMPLIANT);
        return $sql;
    }

    public function isShownEntriesToUser($beneficiaryId = null)
    {
        $isShow = 1;
        if ($beneficiaryId) {
            $beneficiary = BeneficiaryImport::find($beneficiaryId);
            if ($beneficiary) {
                $complaintsSql = $this->getComplaintsDetailsCore($beneficiary);
                $totalComplaints = $complaintsSql->count();

                $isShowSql = $this->getComplaintsDetailsCore($beneficiary);
                $isShowSql->where('feedback.is_show', 1);
                $isShowCount = $isShowSql->count();

                if ($totalComplaints > 0 && $isShowCount > 0) {
                    $percentage = round($totalComplaints * BeneficiaryImport::IS_SHOW_PERCENTAGE / 100);
                    if ($percentage >= $isShowCount) {
                        $isShow = 0;
                    }
                }
            }
        }
        return $isShow;
    }

    public function getFeedbackLevelMobileNumber($beneficiary)
    {
        $escaltion = DB::table('escalation_matrix')
            ->join('escalation_matrix_levels', 'escalation_matrix_levels.escalation_reference_id', 'escalation_matrix.id')
            ->join('awc_master', 'awc_master.id', 'escalation_matrix.awc_id')
            ->where('escalation_matrix_levels.level', 1)
            // ->where('escalation_matrix.scheme_id', $beneficiary->scheme_id)
            // ->where('escalation_matrix.program_id', $beneficiary->program_id)
            // ->where('escalation_matrix.district_id', $beneficiary->district_id)
            ->where('escalation_matrix.awc_id', $beneficiary->awc)
            ->where('escalation_matrix.status', EscalationMatrix::STATUS_ACTIVE)
            ->select('escalation_matrix_levels.mobile', 'escalation_matrix_levels.name', 'awc_master.name as awcName', 'awc_master.code as awcCode')
            ->first();
        return $escaltion;
    }

    public function processFormData($jsonContent, $fieldName, $formValue, HttpRequest $request)
    {

        $typeOfForm = explode('_', $fieldName);
        $typeOfForm = $typeOfForm[0];

        $params = new AppClass();
        $params->jsonContent = $jsonContent;
        $params->fieldName = $fieldName;
        $params->formValue = $formValue;
        $params->typeOfForm = $typeOfForm;

        $formObjects = $this->formService->setFormFieldAttributes($params);

        if ($formObjects->isFile) {

            $files = [];
            if ($request->file($fieldName)) {
                foreach ($request->file($fieldName) as $key => $file) {
                    $fileName = time() . rand(1, 99) . '.' . $file->extension();
                    $file->move(public_path('uploads/feedback-forms'), $fileName);
                    array_push($files, $fileName);
                }
            }

            $formObjects->value = implode(', ', $files);
            $formObjects->type = 'file';
        }

        return $formObjects;
    }

    public function getFeedbackMasters($type = null)
    {
        $master = DB::table('feedback_master')
            ->join('scheme_master', 'scheme_master.id', 'feedback_master.scheme_id')
            ->join('program_master', 'program_master.id', 'feedback_master.program_id')
            ->select('scheme_master.name as scheme', 'program_master.name as program', 'feedback_master.id', 'feedback_master.status')
            ->whereNull('feedback_master.deleted_at')
            ->whereIn('feedback_master.status', FeedbackMaster::STATUS_LIST);
        // ->get();
        return $master;
    }

    public function getBenFeedbackList(HttpRequest $request)
    {
        $authDetails = Auth::user();
        $role = $authDetails->role;
        $userType = $authDetails->user_type;

        $args = $this->reportService->processrequestFilters($request);

        $sql = $this->coreFeedbackSql();
        $sql->leftJoin('agent_master as users', 'users.id', 'feedback.created_by');
        $sql->select(
            'beneficiary_import.status',
            'beneficiary_import.id',
            'beneficiary_import.customer_name',
            'beneficiary_import.mobile_number',
            'feedback.feedback_no',
            'beneficiary_import.skip_reason',
            'beneficiary_import.updated_at as created_at',
            'branch_master.name as branchName',
            'beneficiary_import.description',
            'users.name as createdUser',
            DB::raw('DATE_FORMAT(beneficiary_import.invoice_date, "%d-%m-%Y") as date')
        );
        $sql->whereIn('beneficiary_import.status', [BeneficiaryImport::STATUS_CALL_FEEDBACK_DONE, BeneficiaryImport::STATUS_CALL_SKIPPED]);

        if (!is_null($role)) {

            if (!in_array($role, RoleMaster::SHOW_ALL_FEEDBACK_ROLES) && $userType != User::USER_TYPE_LEVEL_OFFICER) {
                if ($role != RoleMaster::ROLE_AGENT) {
                    $sql->where('feedback.created_by', $authDetails->id);
                }
            }
        }

        if ($args->fromDate && $args->toDate) {
            $fromDate = Carbon::parse($args->fromDate)->format("Y-m-d");
            $toDate = Carbon::parse($args->toDate)->format("Y-m-d");
            $sql->where('beneficiary_import.updated_at', '>=', $fromDate . ' 00:00:00');
            $sql->where('beneficiary_import.updated_at', '<=', $toDate . ' 23:59:59');
        }

        $sql->groupBy('beneficiary_import.id');
        $sql->orderBy('beneficiary_import.id', 'desc');

        return $sql;
    }

    public function getOfficersDetails($authDetails)
    {
        $mobile = $authDetails->email;

        $sql = DB::table('escalation_matrix_levels')
            ->join('escalation_matrix', 'escalation_matrix.id', 'escalation_matrix_levels.escalation_reference_id')
            ->join('block_master', 'block_master.id', 'escalation_matrix.block_id')
            ->join('city_master', 'city_master.id', 'escalation_matrix.district_id');

        if ($authDetails->level == 2) {
            $sql->where('block_master.code', $mobile);
        } else if ($authDetails->level == 3) {
            $sql->where('city_master.code', $mobile);
        } else {
            return [];
        }

        $sql->select('escalation_matrix.scheme_id', 'escalation_matrix.program_id', 'escalation_matrix.district_id', 'escalation_matrix.awc_id');
        $officersDetails = $sql->get();
        return $officersDetails;
    }

    public function getAgentMappedDistrict($agentId)
    {
        $districtId = AgentMapping::where('agent_id', $agentId)->pluck('district_id');
        return $districtId;
    }

    public function coreFeedbackSql()
    {
        $sql = DB::table('beneficiary_import')
            ->leftJoin('feedback', 'feedback.beneficiary_ref_id', 'beneficiary_import.id')
            ->leftJoin('branch_master', 'branch_master.id', 'beneficiary_import.branch_id')
            ->whereNull('beneficiary_import.deleted_at');
        return $sql;
    }

    public function getFeedbackDetails($id)
    {
        $sql = $this->coreFeedbackSql();
        $sql->where('beneficiary_ref_id', $id);
        $sql->select(
            'beneficiary_import.id',
            'beneficiary_import.customer_name',
            'beneficiary_import.mobile_number',
            'beneficiary_import.invoice_date',
            'beneficiary_import.description',
            'feedback.feedback_no',
            'feedback.id as feedbackId',
            'feedback.created_at',
            'users.name as createdUser'
        );
        $sql->join('users', 'users.id', 'feedback.created_by');
        $data = $sql->first();
        if ($data) {
            $data->feedback = FeedbackHistory::select('feedback_history.*', 'call_skip_reason.name as skipReason')
                ->leftJoin('call_skip_reason', 'call_skip_reason.id', 'feedback_history.call_skip_reason')
                ->where('feedback_id', $data->feedbackId)
                ->get();
        }
        return $data;
    }

    public function getFeedbackMasterDetails($id)
    {
        $data = DB::table('feedback_master')
            ->join('scheme_master', 'scheme_master.id', 'feedback_master.scheme_id')
            ->join('program_master', 'program_master.id', 'feedback_master.program_id')
            ->where('feedback_master.id', $id)
            ->select('feedback_master.contents', 'scheme_master.name as scheme', 'program_master.name as program', 'feedback_master.id')
            ->first();
        return $data;
    }

    public function getFileDownloadLink($data)
    {
        foreach ($data->feedback as $feedback) {
            if ($feedback->type == 'file') {
                $html = '';
                $files = explode(',', $feedback->value);
                foreach ($files as $file) {
                    $html .= '<a href="' . route('downloadAttachments', ['type' => 'feedback-forms', 'file' => trim($file)]) . '">' . $file . '</a>,';
                }
                $feedback->value = $html;
            }
        }
    }

    public function isResolvedAuthority($data)
    {
        $authDetails = Auth::user();
        if ($data->resolved_status == Feedback::RESOLVED_STATUS_PENDING && $data->is_compliant == Feedback::IS_COMPLIANT) {
            // $levelAccess = $this->getAuthorityLevelAceess($data, $authDetails);

            if ($this->getAuthorityLevelAceess($data, $authDetails)) {
                return true;
            }

            //for resolving option
            else if (in_array($authDetails->role, RoleMaster::RESOLVE_ROLES)) {
                return true;
            }
        }
    }

    public function officersCore()
    {
        $officer = DB::table('escalation_matrix_levels')
            ->join('escalation_matrix', 'escalation_matrix.id', 'escalation_matrix_levels.escalation_reference_id')
            ->join('block_master', 'block_master.id', 'escalation_matrix.block_id')
            ->join('city_master', 'city_master.id', 'escalation_matrix.district_id')
            ->where('escalation_matrix.status', EscalationMatrix::STATUS_ACTIVE)
            ->whereNull('escalation_matrix.deleted_at');
        return $officer;
    }

    public function getAuthorityLevelAceess($data, $authDetails)
    {

        $esc = $this->officersCore()
            ->where('escalation_matrix_levels.level', $authDetails->level);
        if ($authDetails->level == 3) {
            $esc->where('city_master.code', $authDetails->email);
        } else if ($authDetails->level == 2) {
            $esc->where('block_master.code', $authDetails->email);
        } else {
            return false;
        }
        $officer = $esc->first();

        if ($officer) {
            $sql = $this->officersCore();
            if ($authDetails->level == 3) {
                $sql->where('escalation_matrix.district_id', $officer->district_id);
            } else if ($authDetails->level == 2) {
                $sql->where('escalation_matrix.block_id', $officer->block_id);
            } else {
                return false;
            }

            $awcId = $sql->pluck('escalation_matrix.awc_id')->unique();

            if ($awcId) {
                $awcId = $awcId->toArray();
                if (in_array($data->awc_id, $awcId)) {
                    $preLevelDays = 0;
                    $prevLevel = $this->getNextLevelOfficerId($officer, ($officer->level - 1));
                    if (isset($prevLevel->id)) {
                        $preLevelDays = $prevLevel->days;
                    }

                    $date = Carbon::parse($data->created_at)->format('Y-m-d');
                    $now = Carbon::now()->format('Y-m-d');
                    $dt = Carbon::create($date);
                    $dt2 = Carbon::create($now);
                    // $diffDays = $dt->diffInDaysFiltered(function (Carbon $date) {
                    //     return !$date->isWeekend();
                    // }, $dt2);
                    $diffDays = $dt->diffInDays($dt2);

                    if ($diffDays >= $preLevelDays && $diffDays <= $officer->days) {
                        return true;
                    }
                }
            } else {
                return false;
            }
        }
    }

    public function getNextLevelOfficerId($escalation, $level)
    {
        $nextLevel = $this->officersCore()
            ->where('escalation_matrix_levels.escalation_reference_id', $escalation->escalation_reference_id)
            ->where('escalation_matrix_levels.level', $level)
            ->select('escalation_matrix_levels.*')
            ->first();
        if ($nextLevel) {
            return $nextLevel;
        }
    }

    // public function getAuthorityLevelAceess($data, $authDetails)
    // {

    //     $date = Carbon::parse($data->created_at)->format('Y-m-d');
    //     $now = Carbon::now()->format('Y-m-d');
    //     $dt = Carbon::create($date);
    //     $dt2 = Carbon::create($now);
    //     $diffDays = $dt->diffInDaysFiltered(function (Carbon $date) {
    //         return !$date->isWeekend();
    //     }, $dt2);

    //     $levelDetails = $this->getResolveLevelDays($data, $authDetails);

    //     if ($levelDetails->previousLevelDays == 0 && $levelDetails->currentLevelDays == 0) {
    //         return false;
    //     }

    //     if ($diffDays >= $levelDetails->previousLevelDays && $diffDays <= $levelDetails->currentLevelDays) {
    //         return true;
    //     }
    // }

    public function getAuthorityLevelCoreSql($data, $level = null)
    {
        $sql = DB::table('escalation_matrix_levels')
            ->join('escalation_matrix', 'escalation_matrix.id', 'escalation_matrix_levels.escalation_reference_id')
            // ->where('escalation_matrix.scheme_id', $data->scheme_id)
            // ->where('escalation_matrix.program_id', $data->program_id)
            ->where('escalation_matrix.district_id', $data->district_id)
            ->where('escalation_matrix.status', EscalationMatrix::STATUS_ACTIVE)
            ->whereNull('escalation_matrix.deleted_at');
        if ($level) {
            $sql->where('escalation_matrix_levels.level', '<', $level);
        }
        return $sql;
    }

    public function getResolveLevelDays($data, $authDetails)
    {
        $sql = $this->getAuthorityLevelCoreSql($data)
            ->where('escalation_matrix_levels.mobile', $authDetails->email)
            ->select('escalation_matrix_levels.*');
        $levelDetails = $sql->first();
        $preLevel = null;
        if ($levelDetails) {

            $preLevel = $this->getAuthorityLevelCoreSql($data, $levelDetails->level)
                ->select('escalation_matrix_levels.*')
                ->first();
        }

        $levelData = new AppClass();
        $levelData->currentLevel = isElementHasValueObjectAndReturnZero($levelDetails, 'level');
        $levelData->previousLevel = isElementHasValueObjectAndReturnZero($preLevel, 'level');
        $levelData->previousLevelDays = isElementHasValueObjectAndReturnZero($preLevel, 'days');
        $levelData->currentLevelDays = isElementHasValueObjectAndReturnZero($levelDetails, 'days');

        return $levelData;
    }

    public function resolveFeedback(HttpRequest $request)
    {

        $feedbackId = decrypt($request->benId);
        $beneficiary = DB::table('beneficiary_import')
            ->join('feedback', 'feedback.beneficiary_ref_id', 'beneficiary_import.id')
            ->where('feedback.beneficiary_ref_id', $feedbackId)
            ->select('beneficiary_import.scheme_id', 'beneficiary_import.program_id', 'beneficiary_import.district_id', 'feedback.id')
            ->first();

        if ($beneficiary) {

            // $sql = $this->getAuthorityLevelCoreSql($beneficiary)
            //     ->where('escalation_matrix_levels.mobile', Auth::user()->email)
            //     ->select('escalation_matrix_levels.level');
            // $levelDetails = $sql->first();

            $level = Auth::user()->level;

            $feedback = Feedback::find($beneficiary->id);
            if ($feedback) {
                $feedback->resolved_status = Feedback::RESOLVED_STATUS_RESOLVED;
                $feedback->resolved_remarks = isIssetValue($request, 'resolvedRemarks');
                $feedback->resolve_reason = isIssetValue($request, 'resolveReasons');
                $feedback->resolved_by = Auth::user()->id;
                $feedback->resolved_at = Carbon::now();

                if (Auth::user()->role == RoleMaster::ROLE_AGENT) {
                    $feedback->resolved_level = 1;
                    $feedback->notify_level = 2;
                } else {
                    $feedback->resolved_level = $level;
                    $feedback->notify_level = $level + 1;
                }

                $feedback->save();
                return true;
            }
        }
    }

    public function updateCriticalPoints(HttpRequest $request)
    {
        $checkedItems = $request->has('criticalPoints') ? $request->get('criticalPoints') : null;
        $uniqueId = $request->has('uniqueId') ? $request->get('uniqueId') : null;
        if ($uniqueId) {
            $feedbackMaster = FeedbackMaster::find($uniqueId);
            if ($feedbackMaster) {
                $formFieldContent = json_decode($feedbackMaster->contents, true);
                $updatedJson = $this->updateJsonContent($formFieldContent, $checkedItems);
                if ($updatedJson) {
                    $feedbackMaster->contents = json_encode($updatedJson);
                    $feedbackMaster->save();
                    return true;
                }
            }
        }
        return false;
    }

    public function updateJsonContent($formFieldContent, $checkedItems)
    {
        foreach ($formFieldContent as $key => $jsonContent) {

            $formFieldContent[$key]['criticalPoint'] = false;
            if ($checkedItems && count($checkedItems) > 0) {
                foreach ($checkedItems as $checkedName) {
                    if (isset($jsonContent['name']) && $jsonContent['name'] == $checkedName) {
                        $formFieldContent[$key]['criticalPoint'] = true;
                    }
                }
            }
        }

        return $formFieldContent;
    }

    public function removeHtmlTagsFromString($feedback)
    {
        foreach ($feedback as $list) {
            $list->label = removeHtmlTags($list->label);
        }
    }

    public function saveFeedbackApp(HttpRequest $request)
    {
        try {
            $userId = $request->user()->reference_id;

            $reason = $request->has('callSkipReason') ? $request->get('callSkipReason') : null;
            $callbackOn = $request->has('callbackDateTime') ? $request->get('callbackDateTime') : null;

            if ($userId) {
                DB::beginTransaction();
                // $feedback = Feedback::where('beneficiary_ref_id', $request->customerId)->first();

                if ($reason == CallSkipReason::REASON_CALL_BACK || $reason == CallSkipReason::REASON_LATER_VISIT) {
                    $callbackOn = Carbon::parse($callbackOn);
                }

                $feedbackNo = $this->commonService->generateUniqueScreenNumber(NumberingSystem::SCREEN_FEEDBACK);
                $feedback = new Feedback();
                $feedback->feedback_no = $feedbackNo;
                $feedback->beneficiary_ref_id = $request->customerId;
                $feedback->remarks = isset($request->remarks) ? $request->remarks : null;
                $feedback->call_skip_reason = isset($request->callSkipReason) ? $request->callSkipReason : null;
                $feedback->call_back_date_time = $callbackOn;
                $feedback->call_status = $request->isConnected;
                $feedback->call_type = $request->callType;
                $feedback->campaign_id = isset($request->campaignId) ? $request->campaignId : null;
                $feedback->created_by = $userId;
                $feedback->updated_at = Carbon::now();
                $feedback->updated_by = $userId;
                $feedback->save();

                $today = Carbon::now()->format('Y-m-d');
                //add feedback id into call history table
                $history = CallHistory::select('feedback_id', 'call_recording', 'id')
                    ->where('customer_id', $request->customerId)
                    ->where('agent_id', $userId)
                    ->where('created_at', '>=', $today . ' 00:00:00')
                    ->where('created_at', '<=', $today . ' 23:59:59')
                    ->whereNull('feedback_id')
                    ->orderBy('id', 'desc')
                    ->first();
                if ($history) {
                    $history->feedback_id = $feedback->id;
                    $history->save();

                    //save recording to feeback table
                    $feedback->recordings = $history->call_recording;
                    // $feedback->save();
                }

                $status = BeneficiaryImport::STATUS_CALL_SKIPPED;
                if ($request->isConnected == 1) {
                    $status = BeneficiaryImport::STATUS_CALL_FEEDBACK_DONE;
                }

                //update to beneficiary table
                $data = BeneficiaryImport::find($request->customerId);
                $data->status = $status;
                $data->skip_reason = $reason;
                $data->updated_by = Auth::user()->id;
                $data->callback_on = $callbackOn;
                $data->call_status = null;
                $data->agent_id = $userId;
                $data->save();

                //for just dial table status update
                if ($data->jd_lead_id > 0) {
                    $jdLead = JustDialLeads::select('status', 'remarks', 'id')->find($data->jd_lead_id);
                    if ($jdLead) {
                        $jdLead->status = isset($request->jdStatus) ? $request->jdStatus : null;
                        $jdLead->remarks = isset($request->remarks) ? $request->remarks : null;
                        $jdLead->save();
                    }
                } else if ($data->sap_lead_id > 0) {
                    $enquiry = EnquiryHistory::select('status', 'remarks', 'id')->find($data->sap_lead_id);
                    if ($enquiry) {
                        $enquiry->status = isset($request->SAPStatus) ? $request->SAPStatus : null;
                        $enquiry->remarks = isset($request->remarks) ? $request->remarks : null;
                        $enquiry->save();
                    }
                } else if ($data->crm_lead_id > 0) {
                    $ticket = CRMTickets::select('lead_status', 'id')->find($data->crm_lead_id);
                    if ($ticket) {
                        $ticket->lead_status = isset($request->leadStatus) ? $request->leadStatus : null;
                        $ticket->save();
                    }
                }

                //update cluster details in feedback table
                $cluster = ClusterBranchMapping::select('cluster_master.id')
                    ->join('cluster_master', 'cluster_master.id', 'cluster_branch_mapping.cluster_id')
                    ->whereNull('cluster_master.deleted_at')
                    ->where('cluster_branch_mapping.branch_id', $data->branch_id)
                    ->first();
                $feedback->cluster_id = isset($cluster->id) ? $cluster->id : null;
                $feedback->save();

                DB::commit();
                return $this->commonService->makeSuccessResponse('feedback saved successfully', $feedback);
            } else {
                return $this->commonService->makeErrorResponse('user not found');
            }
        } catch (\Exception $e) {
            DB::rollback();
            Log::info('saveFeedbackApp error : ' . $e->getMessage());
            return $this->commonService->makeErrorResponse('failed to save data');
        }
    }

    function getFeedbackHistory(HttpRequest $request)
    {
        $data = Feedback::select('feedback.call_back_date_time', 'feedback.remarks', 'feedback.call_status', 'call_skip_reason.name as callReason', 'beneficiary_import.description', 'beneficiary_import.invoice_date', 'enquiry_history.amount')
            ->join('beneficiary_import', 'beneficiary_import.id', 'feedback.beneficiary_ref_id')
            ->leftJoin('call_skip_reason', 'call_skip_reason.id', 'feedback.call_skip_reason')
            ->leftJoin('enquiry_history', 'enquiry_history.id', 'beneficiary_import.sap_lead_id')
            ->where('feedback.beneficiary_ref_id', $request->cusId)
            ->get();
        if ($data) {
            foreach ($data as $row) {
                if ($row->call_back_date_time) {
                    $row->call_back_date_time = AppDateTimeFormat($row->call_back_date_time);
                }
            }
        }
        return response()->json($data);
    }
}
