<?php

namespace App\Services\Reports;

use App\Models\Feedback;
use App\Models\Masters\AgentMaster;
use App\Models\Transactions\CallHistory;
use App\Services\Reports\ReportService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use DB;
use stdClass;

class AnalyticsDashboardService
{
    private $reportService;

    public function __construct(ReportService $reportService)
    {
        $this->reportService = $reportService;
    }

    function coreAgentCallDetails($arguments)
    {
        $fromDate = Carbon::parse($arguments->fromDate)->format('Y-m-d');
        $toDate = Carbon::parse($arguments->toDate)->format('Y-m-d');

        $sql = Feedback::join('beneficiary_import', 'beneficiary_import.id', 'feedback.beneficiary_ref_id');
        if ($fromDate && $toDate) {
            $sql->where('feedback.created_at', '>=', $fromDate . ' 00:00:00')
                ->where('feedback.created_at', '<=', $toDate . ' 23:59:59');
        }

        if ($arguments->branch) {
            $sql->where('beneficiary_import.branch_id', $arguments->branch);
        }
        if ($arguments->cluster) {
            $sql->where('feedback.cluster_id', $arguments->cluster);
        }
        return $sql;
    }

    function callHistorySql($args)
    {
        $callSql = $this->coreAgentCallDetails($args);
        $callSql->join('call_history', 'call_history.feedback_id', 'feedback.id')
            ->whereNotNull('call_history.call_recording')
            ->where('call_history.call_recording', '!=', 'None')
            ->select(
                DB::raw("AVG(TIME_TO_SEC(call_history.call_transfer_duration)) as avg_transfer_seconds"),
                DB::raw("AVG(call_history.percentage) as percentage")
            );
        return $callSql;
    }

    function getPreviousDateRange($args)
    {
        $start = Carbon::parse($args->fromDate);
        $end = Carbon::parse($args->toDate);

        $daysDiff = $start->diffInDays($end);
        $previousEnd = $start->copy()->subDay();
        $previousStart = $previousEnd->copy()->subDays($daysDiff);

        $newArgs = clone $args;
        $newArgs->fromDate = $previousStart;
        $newArgs->toDate = $previousEnd;
        return $newArgs;
    }

    function getBranchAnalysis($args)
    {
        $bSql = $this->callHistorySql($args);
        $bSql->select(
            DB::raw("AVG(TIME_TO_SEC(call_history.call_transfer_duration)) as avg_transfer_seconds"),
            DB::raw("ROUND(AVG(call_history.percentage)) as percentage"),
            DB::raw("ROUND(AVG(call_history.sections_covered)) as sections_covered"),
            DB::raw("ROUND(AVG(call_history.sections_partially_covered)) as sections_partially_covered"),
            DB::raw("ROUND(AVG(call_history.sections_missed)) as sections_missed"),
            DB::raw("COUNT(DISTINCT(beneficiary_import.agent_id)) as agentCount"),
            DB::raw("COUNT(feedback.id) as callCount"),
            'branch_master.name as branchName'
        )
            ->join('branch_master', 'branch_master.id', 'beneficiary_import.branch_id')
            ->orderBy('percentage', 'desc')
            ->groupBy('beneficiary_import.branch_id');
        $branchAnalysis = $bSql->take(3)->get();
        return $branchAnalysis;
    }

    function getAgentAnalysis($args)
    {
        $aSql = $this->callHistorySql($args);
        $aSql->select(
            DB::raw("ROUND(AVG(call_history.percentage)) as percentage"),
            DB::raw("COUNT(DISTINCT(beneficiary_import.agent_id)) as agentCount"),
            DB::raw("COUNT(feedback.id) as callCount"),
            'branch_master.name as branchName',
            'agent_master.name as agentName',
        )
            ->join('branch_master', 'branch_master.id', 'beneficiary_import.branch_id')
            ->join('agent_master', 'agent_master.id', 'beneficiary_import.agent_id')
            ->orderBy('percentage', 'desc')
            ->groupBy('beneficiary_import.agent_id');
        $agentAnalysis = $aSql->take(5)->get();
        return $agentAnalysis;
    }

    function getAnalyticsOverview($args)
    {
        $aSql = $this->callHistorySql($args);
        $aSql->join('audio_analytics', 'call_history.id', 'audio_analytics.call_id')
            ->groupBy('audio_analytics.name')
            ->select(
                'audio_analytics.name',
                DB::raw('ROUND(
        CASE 
            WHEN COUNT(CASE WHEN LOWER(audio_analytics.value) = "yes" THEN 1 END) > 
                 COUNT(CASE WHEN LOWER(audio_analytics.value) != "yes" THEN 1 END)
            THEN 
                ((COUNT(CASE WHEN LOWER(audio_analytics.value) = "yes" THEN 1 END) - 
                  COUNT(CASE WHEN LOWER(audio_analytics.value) != "yes" THEN 1 END)) * 100.0) /
                NULLIF(COUNT(audio_analytics.id), 0)
            ELSE 0
        END
    ) as comparison_percentage')
            );

        $analysis = $aSql->get();
        return $analysis;
    }

    function getGraphAnalysis($args)
    {
        $newArgs = clone $args;
        $newArgs->fromDate = null;
        $newArgs->toDate = null;
        $aSql = $this->callHistorySql($newArgs);
        $currentYear = Carbon::parse($args->toDate)->year;
        $previousYear = $currentYear - 1;
        $aSql->select(
            DB::raw('CONCAT(YEAR(call_history.created_at), " Q", QUARTER(call_history.created_at)) as period'),
            DB::raw("CAST(ROUND(AVG(call_history.percentage)) AS UNSIGNED) as percentage")
        )
            ->whereIn(DB::raw('YEAR(call_history.created_at)'), [$previousYear, $currentYear])
            ->groupBy(DB::raw('YEAR(call_history.created_at)'), DB::raw('QUARTER(call_history.created_at)'))
            ->orderBy(DB::raw('YEAR(call_history.created_at)'), 'asc')
            ->orderBy(DB::raw('QUARTER(call_history.created_at)'), 'asc');
        $data = $aSql->get();
        return $data;
    }

    public function getDashboardData(Request $request)
    {
        $args = $this->reportService->processrequestFilters($request);

        $agentsCount = AgentMaster::whereStatus(AgentMaster::STATUS_ACTIVE)->count();
        $coreSql = $this->coreAgentCallDetails($args);
        $callsSql = clone $coreSql;
        $totalCalls = $callsSql->count();

        //getting call analysis data
        $callSql = $this->callHistorySql($args);
        $bSql = clone $callSql;
        $aSql = clone $callSql;
        $history = $callSql->first();

        //getting previous org score
        $newArgs = $this->getPreviousDateRange($args);
        $pSql = $this->callHistorySql($newArgs);
        $historyPre = $pSql->first();

        $avgHandling = 0;
        if ($history && $history->avg_transfer_seconds) {
            $avgHandling = timeFormatting($history->avg_transfer_seconds);
        }

        $dashboardCounts = new stdClass;
        $dashboardCounts->agentsCount = $agentsCount;
        $dashboardCounts->totalCalls = $totalCalls;
        $dashboardCounts->avgHandling = $avgHandling ?? 0;
        $dashboardCounts->orgScore = $history->percentage ? round($history->percentage) : 0;
        $dashboardCounts->orgScorePrevious = ($dashboardCounts->orgScore) - ($historyPre->percentage ? round($historyPre->percentage) : 0);
        if (!$historyPre->percentage) {
            $dashboardCounts->orgScorePrevious = 0;
        }

        //getting branchwise details
        $branchAnalysis = $this->getBranchAnalysis($args);
        // getting agent wise details
        $agentAnalysis = $this->getAgentAnalysis($args);

        //getting graph data
        $graphAnalysis = $this->getGraphAnalysis($args);
        $graphData = new stdClass;
        $graphData->data = $graphAnalysis->pluck('percentage');
        $graphData->categories = $graphAnalysis->pluck('period');

        //analytics overview
        $analyticsOverview = $this->getAnalyticsOverview($args);

        $overviewHtml = view('reports.analytics-overview', compact('analyticsOverview'))->render();
        $branchAnalysisHtml = view('reports.branch-analytics', compact('branchAnalysis'))->render();
        $agentsHtml = view('reports.agents-analytics', compact('agentAnalysis'))->render();

        $returnParams = new stdClass;
        $returnParams->overviewHtml = $overviewHtml;
        $returnParams->branchAnalysisHtml = $branchAnalysisHtml;
        $returnParams->agentsHtml = $agentsHtml;
        $returnParams->graphData = $graphData;
        $returnParams->dashboardCounts = $dashboardCounts;
        return $returnParams;
    }
}
