<?php

namespace App\Services\Reports;

use App\Models\Feedback;
use App\Models\Transactions\BeneficiaryImport;
use App\Models\Transactions\CallHistory;
use App\Services\ExcelExportService;
use App\Services\Reports\ReportService;
use Carbon\Carbon;
use DataTables;
use Illuminate\Http\Request;
use DB;
use Illuminate\Support\Facades\Validator;

class CallsDialledService
{
    public $excelExportService;
    public $reportService;

    public function __construct(ReportService $reportService, ExcelExportService $excelExportService)
    {
        $this->reportService = $reportService;
        $this->excelExportService = $excelExportService;
    }

    public function getData(Request $request, $type = null)
    {

        DB::statement('SET SQL_BIG_SELECTS=1');
        
        $args = $this->reportService->processrequestFilters($request);

        $fromDate = Carbon::parse($args->fromDate)->format('Y-m-d');
        $toDate = Carbon::parse($args->toDate)->format('Y-m-d');

        $sql = Feedback::select(
            'cluster_master.name as clusterName',
            'branch_master.name as branchName',
            'agent_master.name as agentName',
            'designation_master.name as designationName',
            'campaign.title as campaignName',
            'beneficiary_import.customer_name',
            'beneficiary_import.mobile_number',
            'beneficiary_import.description',
            'call_history.call_duration',
            'call_history.call_transfer_duration',
            'call_skip_reason.name as callReason',
            'call_skip_reason.dispositions',
            'agent_master.emp_no as agentId',
            'call_history.id',
            'class_master.name as className',
            'crm_tickets.ticket_no as ticketNo',
            DB::raw('ROUND(call_history.percentage) as percentage'),
            DB::raw('DATE_FORMAT(feedback.created_at, "%d/%m/%Y %h:%i %p") as callDate'),
            DB::raw('DATE_FORMAT(campaign.created_at, "%d/%m/%Y") as campaignDate'),
            DB::raw('CASE WHEN agents_login.agent_id IS NOT NULL THEN "Present" ELSE "Absent" END as attendance'),
            DB::raw('TIMEDIFF(feedback.created_at, call_history.created_at) AS acw'),
            DB::raw('CASE WHEN call_skip_reason.is_answered = 1 THEN "Connected" ELSE "Not Connected" END as callStatus'),
            DB::raw('DATE_FORMAT(feedback.call_back_date_time, "%d/%m/%Y") as callbackDate'),
            DB::raw('DATE_FORMAT(feedback.call_back_date_time, "%h:%i %p") as callbackTime'),
            DB::raw('DATE_FORMAT(agents_login.login_in, "%h:%i %p") as login_in'),
            DB::raw('DATE_FORMAT(agents_login.log_out, "%h:%i %p") as log_out'),
            DB::raw('CONCAT(DATE_FORMAT(call_history.call_date, "%d/%m/%Y"), " ", DATE_FORMAT(call_history.call_time, " %h:%i %p")) as callDateTime'),
            BeneficiaryImport::getSourceCaseQuery(),
            // DB::raw('(CASE WHEN beneficiary_import.source = 1 THEN "Tele Calling" WHEN beneficiary_import.source = 2 THEN "Just Dial" WHEN beneficiary_import.source = 3 THEN "CRM" WHEN beneficiary_import.source = 4 THEN "Social Media" WHEN beneficiary_import.source = 5 THEN "SAP" WHEN beneficiary_import.source = 6 THEN "Vendor Campaign" ELSE "N/A" END) as source'),
            'call_history.call_recording as recordings',
            'feedback.id as feedbackID'
        )
            ->leftJoin('call_history', 'feedback.id', 'call_history.feedback_id')
            ->leftJoin('beneficiary_import', 'beneficiary_import.id', 'feedback.beneficiary_ref_id')
            ->leftJoin('agent_master', 'agent_master.id', 'beneficiary_import.agent_id')
            ->leftJoin('designation_master', 'designation_master.id', 'agent_master.designation')
            ->leftJoin('agents_login', function ($join) use ($args) {
                $join->on('feedback.created_by', '=', 'agents_login.agent_id')
                    ->on(DB::raw('DATE(feedback.created_at)'), '=', 'agents_login.date')
                    ->whereBetween('agents_login.date', [$args->fromDate, $args->toDate]);
            })
            ->leftJoin('cluster_master', 'cluster_master.id', 'feedback.cluster_id')
            ->leftJoin('branch_master', 'branch_master.id', 'agent_master.branch_id')
            ->leftJoin('campaign', 'campaign.id', 'feedback.campaign_id')
            ->leftJoin('call_skip_reason', 'call_skip_reason.id', 'feedback.call_skip_reason')
            ->leftJoin('class_master','class_master.id','agent_master.class')
            ->leftJoin('crm_tickets','crm_tickets.id','beneficiary_import.crm_lead_id')
            ->where('feedback.created_at', '>=', $fromDate . ' 00:00:00')
            ->where('feedback.created_at', '<=', $toDate . ' 23:59:59')
            ->groupBy('feedback.id');
        if ($args->branch && $args->branch != 'All') {
            $sql->where('agent_master.branch_id', $args->branch);
        }
        if ($args->agent && $args->agent != 'All') {
            $sql->where('beneficiary_import.agent_id', $args->agent);
        }
        if ($args->cluster && $args->cluster != 'All') {
            $sql->where('feedback.cluster_id', $args->cluster);
        }
        if ($args->source && $args->source != 'All') {
            $sql->where('beneficiary_import.source', $args->source);
        }

        if ($type == 'excel') {
            return $sql;
        }

        $ret = Datatables::of($sql)
            ->addIndexColumn()
            ->addColumn('downloadRecordings', function ($row) {

                $validator = Validator::make(['url' => $row->recordings], [
                    'url' => 'required|url',
                ]);

                $btn = '';
                if (!$validator->fails()) {
                    $btn = '<a href="' . $row->recordings . '"><i class="fe fe-download attachment_icon"></i></a>';
                }
                return $btn;
            })
            ->filter(function ($query) use ($request) {
                // Capture the global search term
                $searchTerm = $request->input('search.value');

                if ($searchTerm) {
                    $query->where(function ($query) use ($searchTerm) {
                        $query->orWhere('cluster_master.name', 'like', "%{$searchTerm}%")
                            ->orWhere('branch_master.name', 'like', "%{$searchTerm}%")
                            ->orWhere('agent_master.name', 'like', "%{$searchTerm}%")
                            ->orWhere('designation_master.name', 'like', "%{$searchTerm}%")
                            ->orWhere('campaign.title', 'like', "%{$searchTerm}%")
                            ->orWhere('beneficiary_import.customer_name', 'like', "%{$searchTerm}%")
                            ->orWhere('beneficiary_import.mobile_number', 'like', "%{$searchTerm}%")
                            ->orWhere(DB::raw('DATE_FORMAT(feedback.created_at, "%d/%m/%Y %h:%i %p")'), 'like', "%{$searchTerm}%")
                            ->orWhere(DB::raw('DATE_FORMAT(campaign.created_at, "%d/%m/%Y")'), 'like', "%{$searchTerm}%")
                            ->orWhere(DB::raw('TIMEDIFF(feedback.created_at, call_history.created_at)'), 'like', "%{$searchTerm}%")
                            ->orWhere('call_skip_reason.name', 'like', "%{$searchTerm}%");
                    });
                }
            })
            ->filterColumn('callDate', function ($query, $keyword) {
                $formattedKeyword = $this->reportService->formatCreatedDateFromDt($keyword);
                $query->whereRaw("SUBSTRING(feedback.created_at, 1, 16) = ?", [$formattedKeyword]);
            })
            ->filterColumn('attendance', function ($query, $keyword) {
                $keyword = $this->reportService->formatDtKeyword($keyword);
                if ($keyword == 'Present') {
                    $query->whereNotNull('agents_login.agent_id');
                } else {
                    $query->whereNull('agents_login.agent_id');
                }
            })
            ->filterColumn('agents_login.login_in', function ($query, $keyword) {
                $keyword = $this->reportService->formatDtKeyword($keyword);
                $time = Carbon::parse($keyword)->format('H:i');
                $query->whereRaw("SUBSTRING(agents_login.login_in, 1, 5) = ?", [$time]);
            })
            ->filterColumn('agents_login.log_out', function ($query, $keyword) {
                $keyword = $this->reportService->formatDtKeyword($keyword);
                $time = Carbon::parse($keyword)->format('H:i');
                $query->whereRaw("SUBSTRING(agents_login.log_out, 1, 5) = ?", [$time]);
            })
            ->filterColumn('campaignDate', function ($query, $keyword) {
                $keyword = $this->reportService->formatDateFromDt($keyword);
                $query->whereDate('campaign.created_at', $keyword);
            })
            ->filterColumn('callDateTime', function ($query, $keyword) {
                $formattedKeyword = $this->reportService->formatCreatedDateFromDt($keyword);
                $query->whereRaw("CONCAT(call_date,' ',SUBSTRING(call_time,1,5)) = ?", $formattedKeyword);
            })
            ->filterColumn('acw', function ($query, $keyword) {
                $formattedKeyword = $this->reportService->formatDtKeyword($keyword);
                $query->where(DB::raw('TIMEDIFF(feedback.created_at, call_history.created_at)'), $formattedKeyword);
            })
            ->filterColumn('callStatus', function ($query, $keyword) {
                $formattedKeyword = $this->reportService->formatDtKeyword($keyword);
                if ($formattedKeyword == 'Connected') {
                    $query->where('call_skip_reason.is_answered', 1);
                } else {
                    $query->where('call_skip_reason.is_answered', '!=', 1);
                }
            })
            ->filterColumn('callbackDate', function ($query, $keyword) {
                $formattedKeyword = $this->reportService->formatDateFromDt($keyword);
                $query->whereDate('feedback.call_back_date_time', $formattedKeyword);
            })
            ->filterColumn('callbackTime', function ($query, $keyword) {
                $formattedKeyword = $this->reportService->formatDtKeyword($keyword);
                $time = Carbon::parse($formattedKeyword)->format('H:i');
                $query->where(DB::raw("SUBSTRING(feedback.call_back_date_time, 12, 5)"), $time);
            });


        $ret->addColumn('action', function ($row) {

            $btn = '<div class="dropdown">
                <button aria-expanded="false" aria-haspopup="true" class="btn btn-sm ripple btn-primary dropdown-toggle" data-toggle="dropdown" id="dropdownMenuButton" type="button">Action <i class="fas fa-caret-down ml-1"></i></button>
                <div  class="dropdown-menu tx-13">';
            $btn .= '<a class="dropdown-item" href="' . route('view-transcript', encrypt($row->id)) . '">View Transcript</a>';
            $btn .= '</div>
                        </div>';

            return $btn;
        })

            ->rawColumns(['downloadRecordings', 'action']);

        return $ret->make(true);
    }

    function getCallsDialledReportExcel(Request $request)
    {
        $sql = $this->getData($request, 'excel');
        $data = $sql->get();
        $args = $this->reportService->processrequestFilters($request);
        $params = clone $args;
        $params->type = 'calldialled-report-excel';
        $params->branchLabel = $this->reportService->getBranchById($args->branch);
        $params->agentLabel = $this->reportService->getAgentById($args->agent);
        $params->clusterLabel = $this->reportService->getClusterById($args->cluster);
        return $this->excelExportService->exportExcel($params, $data);
    }

    function viewTranscript($id)
    {
        $data = CallHistory::find($id);
        if ($data) {
            $data->transcript = json_decode($data->process_response);
            return $data;
        }
    }
}
