<?php

namespace App\Services\Reports;

use App\Models\AgentsLogin;
use App\Models\ApplicationSettings;
use App\Services\ExcelExportService;
use App\Services\Reports\ReportService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use DB;

class NocallAgentsService
{
    protected $reportService;
    protected $excelExportService;

    public function __construct(ReportService $reportService, ExcelExportService $excelExportService)
    {
        $this->reportService = $reportService;
        $this->excelExportService = $excelExportService;
    }

    public function getNoCallAgents(Request $request)
    {
        $args = $this->reportService->processrequestFilters($request);
        $today = Carbon::now()->format('Y-m-d');
        $timeLimit = ApplicationSettings::where('setting_name', 'no_call_agent')->value('setting_value');
        DB::statement('SET SQL_BIG_SELECTS=1');
        $query = AgentsLogin::select('agents_login.id as agentId', 'feedback.id as feedbackId')
            ->leftJoin('feedback', function ($join) use ($timeLimit) {
                $fromDate = Carbon::now()->startOfDay()->format('Y-m-d H:i:s');
                $toDate = Carbon::now()->setTimeFromTimeString($timeLimit)->format('Y-m-d H:i:s');

                $join->on('feedback.created_by', '=', 'agents_login.agent_id')
                    ->whereBetween('feedback.created_at', [$fromDate, $toDate]);
            })
            ->join('agent_master', 'agent_master.id', 'agents_login.agent_id')
            ->join('branch_master', 'branch_master.id', 'agent_master.branch_id')
            ->leftJoin('cluster_branch_mapping', 'cluster_branch_mapping.branch_id', 'agent_master.branch_id')
            ->leftJoin('cluster_master', 'cluster_master.id', 'cluster_branch_mapping.cluster_id')
            ->where('agents_login.date', '>=', $today)
            ->where('agents_login.date', '<=', $today)
            ->whereNull('feedback.id')
            ->select(
                'cluster_master.name as clusterName',
                'branch_master.name as branchName',
                'agent_master.name as agentName',
                'agent_master.emp_no as agentId',
                DB::raw('DATE_FORMAT(CURDATE(), "%d-%m-%Y") as Date')
            );

        if ($args->branch && $args->branch != 'All') {
            $query->where('branch_master.id', $args->branch);
        }
        if ($args->agent && $args->agent != 'All') {
            $query->where('agent_master.id', $args->agent);
        }
        if ($args->cluster && $args->cluster != 'All') {
            $query->where('cluster_master.id', $args->cluster);
        }

        $query->orderBy('agent_master.name');
        $query->groupBy('agent_master.id');
        return $query;
    }

    function getReportExcel(Request $request)
    {
        $sql = $this->getNoCallAgents($request);
        $data = $sql->get();
        $args = $this->reportService->processrequestFilters($request);
        $params = clone $args;
        $params->type = 'nocalls-report-excel';
        $params->branchLabel = $this->reportService->getBranchById($args->branch);
        $params->agentLabel = $this->reportService->getAgentById($args->agent);
        $params->clusterLabel = $this->reportService->getClusterById($args->cluster);
        return $this->excelExportService->exportExcel($params, $data);
    }
}
