<?php

namespace App\Services\Reports;

use App\Entities\ProgramwiseReportEntity;
use App\Models\FeedbackMaster;
use App\Services\Reports\ReportService;
use App\Utilities\AppClass;
use DB;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;

class ProgramwiseService
{

    private $reportService;
    private $result;
    private $feedbackMaster;

    const ARRAY_FIELD_TYPES = ['checkbox-group', 'select', 'radio-group'];

    public function __construct(ReportService $reportService)
    {
        $this->reportService = $reportService;
        $this->result = new Collection();
        $this->feedbackMaster = new Collection();
    }

    public function getProgramwisereport(Request $request)
    {
        $args = $this->reportService->processrequestFilters($request);
        $sql = $this->getReports($args);
        $report = $sql->get();

        $this->feedbackMaster = FeedbackMaster::whereStatus(FeedbackMaster::STATUS_ACTIVE)->get();

        if ($report) {
            foreach ($report as $resultRow) {
                $parent = $this->createParentRow($resultRow);
                $node = $this->createChildRow($resultRow, $parent, $args);
                if ($node->feedbackCount > 0) {
                    $this->result->add($node);
                }
            }
        }

        return $this->result;
    }

    public function getReports($args)
    {
        $sql = $this->reportService->coreFeedbackSqlWithArgs($args);
        $sql->select(
            'scheme_master.name as scheme', 'program_master.name as program', 'beneficiary_import.scheme_id', 'beneficiary_import.program_id', 'beneficiary_import.district_id'
        );
        $sql->groupBy('beneficiary_import.scheme_id', 'beneficiary_import.program_id');
        return $sql;
    }

    public function createParentRow($result)
    {
        $node = new ProgramwiseReportEntity();
        $node->schemeName = $result->scheme;
        $node->programName = removeHtmlTags($result->program);
        $node->schemeId = $result->scheme_id;
        $node->programId = $result->program_id;
        return $node;
    }

    public function getFeedbackResults($result, $args)
    {

        $sql = $this->reportService->coreFeedbackSql();
        if ($result->scheme_id > 0) {
            $sql->where('beneficiary_import.scheme_id', $result->scheme_id);
        }
        if ($result->program_id > 0) {
            $sql->where('beneficiary_import.program_id', $result->program_id);
        }
        if ($args->district > 0) {
            $sql->where('beneficiary_import.district_id', $args->district);
        }

        if ($args->fromDate != '' && $args->toDate != '') {
            $sql->whereDate('feedback.created_at', '>=', $args->fromDate);
            $sql->whereDate('feedback.created_at', '<=', $args->toDate);
        }

        $sql->join('feedback_master_form_field', 'feedback_master_form_field.feedback_ref_id', 'feedback.id');

        $sql->select('feedback_master_form_field.others_values',
            DB::raw('COUNT(feedback_master_form_field.value) as ansConut'),
        );

        if (isset($args->fieldType) && (in_array($args->fieldType, self::ARRAY_FIELD_TYPES))) {
            $sql->where('feedback_master_form_field.value', 'LIKE', "%{$args->answer}%");
        } else {
            $sql->where('feedback_master_form_field.value', $args->answer);
        }
        $sql->where('feedback_master_form_field.name', $args->question);

        $sql->orderBy('feedback_master_form_field.id');

        if (isset($args->others) && $args->others) {
            $sql->whereNotNull('feedback_master_form_field.others_values');
            $sql->groupBy('feedback_master_form_field.others_values');
            $data = $sql->get();
        } else {
            $data = $sql->first();
        }

        return $data;
    }

    public function appendFieldNameToName($jsonContent)
    {
        $fieldType = $jsonContent['type'];
        switch ($fieldType) {
            case 'checkbox-group':
                $html = 'checkbox-group_' . $jsonContent['name'];
                break;
            case 'date':
                $html = 'date_' . $jsonContent['name'];
                break;
            case 'number':
                $html = 'number_' . $jsonContent['name'];
                break;
            case 'text':
                $html = 'text_' . $jsonContent['name'];
                break;
            case 'radio-group':
                $html = 'radio-group_' . $jsonContent['name'];
                break;
            case 'select':
                $html = 'select_' . $jsonContent['name'];
                break;
            case 'textarea':
                $html = 'textarea_' . $jsonContent['name'];
                break;
        }
        return $html;
    }

    public function getQuestionsAndAnswers($result)
    {
        $questions = new Collection();

        foreach ($this->feedbackMaster as $feedback) {
            if ($feedback->scheme_id == $result->scheme_id && $feedback->program_id == $result->program_id) {
                $json = json_decode($feedback->contents, true);
                foreach ($json as $value) {
                    if (isset($value['criticalPoint']) && $value['criticalPoint']) {
                        $name = $this->appendFieldNameToName($value);
                        $question = new AppClass();
                        $question->label = trim($value['label']);
                        $question->name = trim($name);
                        $question->class = '';
                        $question->fieldType = $value['type'];
                        $question->otherOption = isset($value['other']) ? $value['other'] : null;
                        $question->answers = new Collection();

                        if (isset($value['values']) && count($value['values']) > 0) {

                            foreach ($value['values'] as $options) {
                                $answer = new AppClass();
                                $answer->label = trim($options['label']);
                                $answer->count = 0;
                                $answer->percentage = 0;
                                $answer->others = false;
                                $question->answers->add($answer);
                            }
                        }

                        $questions->add($question);

                    }
                }
            }

        }

        return $questions;
    }

    public function createChildRow($result, $parent, $args)
    {
        $ansCount = new Collection();
        $parent->questions = $this->getQuestionsAndAnswers($result);

        if ($parent->questions && count($parent->questions) > 0) {

            $sno = 0;

            foreach ($parent->questions as $question) {

                if ($sno > 4) {
                    $sno = 0;
                }
                $question->class = $this->getStyleClass($sno);
                $args->question = $question->name;
                $args->fieldType = $question->fieldType;
                $sno++;

                if ($question->answers) {

                    foreach ($question->answers as $answer) {

                        $args->answer = $answer->label;
                        $args->others = $answer->others;

                        $answerDetails = $this->getFeedbackResults($result, $args);

                        if ($answerDetails) {
                            $answer->count = $answerDetails->ansConut;
                        }

                    }

                    //other option answers

                    if ($question->otherOption) {
                        $args->answer = 'Others';
                        $args->others = true;

                        $answerDetails = $this->getFeedbackResults($result, $args);

                        if ($answerDetails && count($answerDetails) > 0) {

                            foreach ($answerDetails as $otherAns) {
                                $others = new AppClass();
                                $others->label = 'Others - ' . $otherAns->others_values;
                                $others->count = $otherAns->ansConut;
                                $others->percentage = 0;
                                $others->others = true;
                                $question->answers->add($others);
                            }
                        }

                    }

                    $totalAnswers = $question->answers->sum('count');
                    $ansCount->add($totalAnswers);
                    foreach ($question->answers as $answer) {
                        if ($answer->count > 0 && $totalAnswers > 0) {
                            $answer->percentage = round($answer->count / $totalAnswers * 100);
                        }
                    }
                }
            }
        }
        $parent->feedbackCount = $ansCount->max();
        return $parent;

    }

    public function getStyleClass($index)
    {
        $class = [
            'bg-primary',
            'bg-success',
            'bg-danger',
            'bg-warning',
            'bg-default',
        ];
        return $class[$index];
    }

    public function getDetailedReport(Request $request)
    {
        
        $sql = $this->reportService->coreFeedbackSqlWithArgs($request);
        $sql->join('feedback_master_form_field', 'feedback_master_form_field.feedback_ref_id', 'feedback.id')
            ->join('city_master', 'city_master.id', 'beneficiary_import.district_id')
            ->join('users', 'users.id', 'feedback.created_by');

        if ($request->question) {
            $sql->where('feedback_master_form_field.name', $request->question);
        }
        if ($request->answer) {
            if (isset($request->others) && $request->others) {
                $otherAns = explode('-', $request->answer);
                $others = isset($otherAns[1]) ? trim($otherAns[1]) : null;
                if ($others) {
                    $sql->where('feedback_master_form_field.others_values', 'LIKE', "{$others}");
                    $sql->where('feedback_master_form_field.value', 'LIKE', "Others");
                } else {
                    $sql->where('feedback_master_form_field.value', 'LIKE', "%{$request->answer}%");
                }
            } else {
                $sql->where('feedback_master_form_field.value', 'LIKE', "%{$request->answer}%");
            }

        }

        $sql->select('program_master.name as program', 'beneficiary_import.status', 'beneficiary_import.id', 'scheme_master.name as scheme', 'city_master.name as district', 'beneficiary_import.customer_name', 'beneficiary_import.mobile_number', 'feedback.feedback_no', 'beneficiary_import.skip_reason', 'feedback.created_at', 'beneficiary_import.id', 'users.name as call_user',
            DB::raw('(CASE WHEN feedback.is_compliant = 1 THEN "YES" ELSE "NO" END) as complaint_registered'),
            DB::raw('(CASE WHEN feedback.recordings IS NOT NULL THEN feedback.recordings ELSE beneficiary_import.recordings END) as recordings')
        );  

        if (isset($request->excel) && $request->excel) {
            $data = $sql->get();
            return $data;
        }

        return $sql;

    }

    public function getDistrictwiseProgramReport(Request $request)
    {
        $request->districtwise = true;
        $sql = $this->getDetailedReport($request);
        $sql->addSelect(
            DB::raw('COUNT(feedback.id) as count')
        );
        $sql->groupBy('beneficiary_import.district_id');
        $data = $sql->get();
        $data->screen = 'programwise-report';
        $html = view('reports.programwise-district-count-report', compact('data'))->render();
        $response = [
            'status' => true,
            'redirect' => null,
            'renderId' => '#districtwise-report-content',
            'html' => $html,
            'tableId' => '#districtwise',
            'export' => true
        ];
        return response()->json($response);
    }

}
