<?php

namespace App\Services\Transactions;

use App\Models\Masters\AgentMapping;
use App\Models\Masters\AgentMaster;
use App\Models\Masters\BranchMaster;
use App\Models\NumberingSystem;
use App\Models\Transactions\BeneficiaryImport;
use App\Models\Transactions\CallMapping;
use App\Models\Transactions\CallMappingItems;
use App\Services\CommonService;
use App\Services\ExcelExportService;
use Illuminate\Http\Request as HttpRequest;
use Illuminate\Support\Facades\Auth;
use DB;
use Log;
use stdClass;

class CallMappingService
{

    private $commonService;
    private $excelExportService;

    public function __construct(CommonService $commonService, ExcelExportService $excelExportService)
    {
        $this->commonService = $commonService;
        $this->excelExportService = $excelExportService;
    }

    public function getData()
    {
        $data = CallMapping::select(
            'call_mapping.*',
            'branch_master.name as branchName',
            DB::raw('(CASE WHEN call_mapping.type = 1 THEN "Auto" ELSE "Manual" END) as mapType'),
            DB::raw('DATE_FORMAT(call_mapping.created_at, "%d-%m-%Y") as date')
        )
            ->join('branch_master', 'branch_master.id', 'call_mapping.branch_id')
            ->orderBy('call_mapping.id', 'desc');
        return $data;
    }

    public function isNew($id = null)
    {
        if ($id) {
            $id = decrypt($id);
            $data = BranchMaster::find($id);
            $data->route = route('call-mapping.update', $id);
            $data->method = 'PUT';
            $data->breadcrumbName = 'Edit';
        } else {
            $data = new BranchMaster();
            $data->route = route('call-mapping.store');
            $data->method = 'POST';
            $data->breadcrumbName = 'Create';
        }
        $data->breadcrumbAction = route('call-mapping.index');
        $data->breadcrumbTitle = 'Call Mapping';

        return $data;
    }

    public function saveData(HttpRequest $request, $id = null)
    {
        try {
            DB::beginTransaction();
            $cusCount = count($request->customerId);
            $agentCount = count($request->agent);
            $perAgent = round($cusCount / $agentCount);
            $splitCustomer = array_chunk($request->customerId, $perAgent);
            $mappingNo = $this->commonService->generateUniqueScreenNumber(NumberingSystem::SCREEN_CALL_MAPPING);
            $mapping = new CallMapping();
            $mapping->mapping_no = $mappingNo;
            $mapping->type = CallMapping::MANUAL_MAPPING;
            $mapping->branch_id = $request->branch;
            $mapping->created_by = Auth::user()->id;
            if ($mapping->save()) {
                foreach ($splitCustomer as $index => $customerChunk) {
                    $agentId = $request->agent[$index];
                    foreach ($customerChunk as $customers) {
                        foreach ($customers as $customerId => $value) {
                            $mappingItem = new CallMappingItems();
                            $mappingItem->mapping_id = $mapping->id;
                            $mappingItem->agent_id = $agentId;
                            $mappingItem->beneficiary_id = $customerId;
                            $mappingItem->save();

                            //update beneficiary table for agent allocation
                            $beneficiary = BeneficiaryImport::find($customerId);
                            if ($beneficiary) {
                                $beneficiary->agent_id = $agentId;
                                $beneficiary->save();
                            }
                        }
                    }
                }
                DB::commit();
                return true;
            }
        } catch (\Exception $e) {
            DB::rollback();
            Log::info('call mapping error : ' . $e->getMessage());
            dd($e);
        }
    }

    function getAgents(HttpRequest $request)
    {
        $agents = AgentMaster::select('id', 'name')
            ->where('branch_id', $request->branchId)
            ->get();
        return response()->json($agents);
    }

    function getProducts(HttpRequest $request)
    {
        $products = BeneficiaryImport::select('description')
            ->where('branch_id', $request->branchId)
            ->groupBy('description')
            ->get();
        return response()->json($products);
    }

    function getCustomerList(HttpRequest $request)
    {
        $customers = BeneficiaryImport::select(
            'id',
            'customer_name',
            'mobile_number',
            'description',
            DB::raw('DATE_FORMAT(invoice_date, "%d-%m-%Y") as date')
        )
            ->where('branch_id', $request->branchId)
            ->where('status', BeneficiaryImport::STATUS_ACTIVE)
            ->whereNull('agent_id')
            ->get();
        return response()->json($customers);
    }

    function showData($id)
    {
        $mapping = CallMapping::select(
            'call_mapping.mapping_no',
            'branch_master.name as branchName',
            'users.name as createdUser',
            DB::raw('DATE_FORMAT(call_mapping.created_at, "%d-%m-%Y") as date')
        )
            ->join('branch_master', 'branch_master.id', 'call_mapping.branch_id')
            ->leftJoin('users', 'users.id', 'call_mapping.created_by')
            ->find($id);
        if ($mapping) {
            $mapping->items = CallMappingItems::select('agent_master.name as agentName', 'beneficiary_import.customer_name', 'beneficiary_import.mobile_number', 'beneficiary_import.description')
                ->join('agent_master', 'agent_master.id', 'call_mapping_items.agent_id')
                ->join('beneficiary_import', 'beneficiary_import.id', 'call_mapping_items.beneficiary_id')
                ->where('call_mapping_items.mapping_id', $id)
                ->get();
        }
        return $mapping;
    }

    function exportExcel($id)
    {
        $data = $this->showData($id);
        if ($data) {
            $data->unAssigned = CallMappingItems::select('beneficiary_import.customer_name', 'beneficiary_import.mobile_number', 'beneficiary_import.description', DB::raw('"Unassigned" as agentName'))
                ->join('beneficiary_import', 'beneficiary_import.id', 'call_mapping_items.beneficiary_id')
                ->where('call_mapping_items.mapping_id', $id)
                ->whereNull('call_mapping_items.agent_id')
                ->get();
            $params = new stdClass;
            $params->type = 'call-mapping';
            return $this->excelExportService->exportExcel($params, $data);
        }
    }
}
